/*
 * Copyright (c) 2012, Metron, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Metron, Inc. nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL METRON, INC. BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.metsci.glimpse.painter.info;

import com.metsci.glimpse.context.GlimpseBounds;
import com.metsci.glimpse.context.GlimpseContext;
import com.metsci.glimpse.support.color.GlimpseColor;

/**
 * Displays an estimate of the current number of times
 * the {@link com.metsci.glimpse.canvas.GlimpseCanvas}
 * is being redrawn per second.
 *
 * @author ulman
 *
 */
public class FpsPainter extends SimpleTextPainter
{
    private static float[] normColor = new float[] { 0.5f, 1.0f, 0.5f, 1.0f };
    private static float[] warnColor = new float[] { 1.0f, 0.5f, 0.5f, 1.0f };

    private float lastFpsEstimate = -1;
    private long timeOfLastCounterReset = -1;
    private float frameCount = 0;

    public FpsPainter( )
    {
        super( );

        setFont( 12, true, false );
        setBackgroundColor( GlimpseColor.getGray( 0.5f ) );
        setPaintBackground( true );
    }

    private final void tickTock( )
    {
        frameCount++;
        long currentTime = System.currentTimeMillis( );

        if ( timeOfLastCounterReset < 0 )
        {
            timeOfLastCounterReset = System.currentTimeMillis( );
            frameCount = 0;
        }
        else if ( currentTime - timeOfLastCounterReset >= 1000 || frameCount >= 10 )
        {
            lastFpsEstimate = frameCount / ( currentTime - timeOfLastCounterReset ) * 1000f;
            timeOfLastCounterReset = currentTime;
            frameCount = 0;
        }
    }

    @Override
    protected void paintTo( GlimpseContext context, GlimpseBounds bounds )
    {
        tickTock( );

        String text = "FPS: Measuring...";
        setColor( GlimpseColor.getBlack( ) );

        if ( lastFpsEstimate > 0 )
        {
            text = String.format( "FPS:%4.0f", lastFpsEstimate );

            if ( lastFpsEstimate > 30 )
            {
                setColor( normColor );
            }
            else
            {
                setColor( warnColor );
            }
        }

        setText( text );

        super.paintTo( context, bounds );
    }
}
